/*  symbimazeappview.cpp
 *
 *  Rendering engine
 *
 *  (c) 2009 Anton Olkhovik <ant007h@gmail.com>
 *
 *  This file is part of SymbiMaze (port of Mokomaze) - labyrinth game.
 *
 *  SymbiMaze is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  SymbiMaze is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SymbiMaze.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "symbimazeapplication.h"

#include <SymbiMazeAppImages.mbg>
#include <aknview.h>
#include <AknUtils.h>
#include <akndef.h>
#include <eikapp.h>
#include <eikappui.h>

#include <aknmessagequerydialog.h>
#include <SymbiMaze.rsg>

//#include "bitmapmethods.h"
#include "IconFileProvider.h"
#include "SymbiMazeAppView.h"
#include "sprite.h"
#include "game.h"
#include "JsonParser.h"

#define PROC_ACC_DATA_INTERVAL 14 //10*1.4

#define TRANS_X shiftImg.iX + scaleImg*
#define TRANS_Y shiftImg.iY + scaleImg*

// ============================ MEMBER FUNCTIONS ===============================

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::CSymbiMazeAppView()
// C++ default constructor can NOT contain any code, that might leave.
// -----------------------------------------------------------------------------
//
CSymbiMazeAppView::CSymbiMazeAppView()
    : iOffScreenBitmap(NULL)
    , iOffScreenBitmapDevice(NULL)
    , iOffScreenBitmapGc(NULL)
    {
    // No implementation required
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::NewL()
// Two-phased constructor.
// -----------------------------------------------------------------------------
//
CSymbiMazeAppView* CSymbiMazeAppView::NewL( const TRect& aRect )
    {
    CSymbiMazeAppView* self = NewLC( aRect );
    CleanupStack::Pop( self );
    return self;
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::NewLC()
// Two-phased constructor.
// -----------------------------------------------------------------------------
//
CSymbiMazeAppView* CSymbiMazeAppView::NewLC( const TRect& aRect )
    {
    CSymbiMazeAppView* self = new ( ELeave ) CSymbiMazeAppView;
    CleanupStack::PushL( self );
    self->ConstructL( aRect );
    return self;
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::ConstructL()
// Symbian 2nd phase constructor can leave.
// -----------------------------------------------------------------------------
//
void CSymbiMazeAppView::ConstructL( const TRect& aRect )
    {
    // Create a window for this application view
    CreateWindowL();

    InitMaze();
    
	iAppRect = aRect;
	TSize resSize;
	AknLayoutUtils::LayoutMetricsSize(AknLayoutUtils::EScreen, resSize);
	iFullRect.SetRect(0,0,resSize.iWidth, resSize.iHeight);
	TReal kw = (TReal)iFullRect.Width() / maze->qt_game_config.wnd_w;  
	TReal kh = (TReal)iFullRect.Height() / maze->qt_game_config.wnd_h;
	if (kw<kh)
		scaleImg = kw;
	else
		scaleImg = kh;
	fieldSize.iWidth = (TInt)(maze->qt_game_config.wnd_w * scaleImg);
	fieldSize.iHeight = (TInt)(maze->qt_game_config.wnd_h * scaleImg);
	shiftImg.iX = (iFullRect.Width() - fieldSize.iWidth) / 2;
	shiftImg.iY = (iFullRect.Height() - fieldSize.iHeight) / 2;
	ballr = (TInt)(maze->qt_game_config.ball_r * scaleImg);
	holer = (TInt)(maze->qt_game_config.hole_r * scaleImg);
	
	iUsingOffScreenBitmap = EFalse;

    // Set the window's size and position
    SetRect( iFullRect );

    // Load in the bitmap images from the multi bitmap file

    // create and open file server session
    User::LeaveIfError(iFsSession.Connect());
    iFsSession.ShareProtected();
    TFileName filePath;

    // set path of the bitmap file
    User::LeaveIfError(iFsSession.PrivatePath(filePath));

    // append the MBM file name to the private path
    filePath.Append(KMultiBitmapFilename);

    // insert the drive to the private path
    TParsePtrC parse((CEikonEnv::Static()->EikAppUi()->Application())->AppFullName());
    filePath.Insert(0, parse.Drive());

    // with the help of icon-provider AknIconUtils can get get the file
    // from this application's private folder
	iIconProvider = CIconFileProvider::NewL( iFsSession, filePath );

    iBackgroundImage = AknIconUtils::CreateIconL(*iIconProvider,
                                                 EMbmSymbimazeappimagesDesk);
	// Load sprite bitmap and mask with a single call
	AknIconUtils::CreateIconL(iBallImage,
	                          iBallMask,
	                          *iIconProvider,
	                          EMbmSymbimazeappimagesBall,
	                          EMbmSymbimazeappimagesBall_mask);

	AknIconUtils::CreateIconL(iHoleImage,
	                          iHoleMask,
	                          *iIconProvider,
	                          EMbmSymbimazeappimagesHole,
	                          EMbmSymbimazeappimagesHole_mask);
	
	AknIconUtils::CreateIconL(iFinImage,
	                          iHoleMask,
	                          *iIconProvider,
	                          EMbmSymbimazeappimagesFin,
	                          EMbmSymbimazeappimagesHole_mask);
	
    // This call preserves SVG-T vector data in memory so that the data is not needed to
    // load from the resource when the icon is subsequently scaled.
//    AknIconUtils::PreserveIconData(iSpriteImage2);

    ScaleImages();

    // Create a periodic timer but don't start it yet
    iPeriodicTimer = CPeriodic::NewL( CActive::EPriorityStandard );

    // Create the array of sprite pointers
    iSprites = new ( ELeave ) CArrayPtrFlat<CSprite> ( 1 );

    SetUpSpritesL();
    
    iSelector=CRemConInterfaceSelector::NewL();
    iTarget = CRemConCoreApiTarget::NewL( *iSelector, *this );
    iSelector->OpenTargetL();
    
    // Activate the window, which makes it ready to be drawn
    ActivateL();

    StartGame();    
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::~CSymbiMazeAppView()
// Destructor.
// -----------------------------------------------------------------------------
//
CSymbiMazeAppView::~CSymbiMazeAppView()
    {
	if ( iPeriodicTimer )
		{
		// Stop the periodic timer
		iPeriodicTimer->Cancel();
		}
	delete iPeriodicTimer;
	iPeriodicTimer = NULL;

	delete iOffScreenBitmapGc;
	iOffScreenBitmapGc = NULL;

	delete iOffScreenBitmapDevice;
	iOffScreenBitmapDevice = NULL;

	delete iOffScreenBitmap;
	iOffScreenBitmap = NULL;

	delete iBackgroundImage;
	iBackgroundImage = NULL;

	if ( iSprites )
		{
		iSprites->ResetAndDestroy();
		delete iSprites;
		iSprites = NULL;
		}

	delete iBallImage;
	iBallImage = NULL;
	delete iBallMask;
	iBallMask = NULL;

	delete iHoleImage;
	iHoleImage = NULL;
	delete iHoleMask;
	iHoleMask = NULL;

	delete iFinImage;
	iFinImage = NULL;
	delete iFinMask;
	iFinMask = NULL;

	iFsSession.Close();
	delete iIconProvider;
	
	//----------

	// Don't delete iTarget, it is owned by the interface selector.
	delete iSelector;
	
	delete maze;
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::SetUpSpritesL()
// Sets up sprites.
// -----------------------------------------------------------------------------
//
void CSymbiMazeAppView::SetUpSpritesL()
	{
    ASSERT( iBallImage );
    ASSERT( iHoleImage );
    ASSERT( iFinImage );
	ASSERT( iSprites );

	iBallSprite = CSprite::NewLC( KInitialXSpeed,
	                         KInitialYSpeed+1,
	                         Rect().iTl,
	                         iBallImage,
	                         iBallMask );
	iSprites->AppendL( iBallSprite );
	CleanupStack::Pop( iBallSprite );

	iHoleSprite = CSprite::NewLC( KInitialXSpeed,
	                         -KInitialYSpeed+1,
	                         Rect().iTl + iHoleImage->SizeInPixels(),
	                         iHoleImage,
	                         iHoleMask );
	iSprites->AppendL( iHoleSprite );
	CleanupStack::Pop( iHoleSprite );
	
	iFinSprite = CSprite::NewLC( KInitialXSpeed,
	                         -KInitialYSpeed+1,
	                         Rect().iTl + iHoleImage->SizeInPixels(),
	                         iFinImage,
	                         iHoleMask );
	iSprites->AppendL( iFinSprite );
	CleanupStack::Pop( iFinSprite );
	}

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::Draw()
// Draws the display.
// -----------------------------------------------------------------------------
//
void CSymbiMazeAppView::Draw( const TRect& /*aRect*/ ) const
    {
	// The system GC will already be activated when this function is called
	// by the framework
	UpdateDisplay();
    }

// -----------------------------------------------------------------------------
// Handle resource change event, here it is the event that occurs
// when screen mode changes.
// -----------------------------------------------------------------------------
void CSymbiMazeAppView::HandleResourceChange(TInt aType, const TRect& aRect)
    {
	CCoeControl::HandleResourceChange(aType);
/*
	if( aType == KEikDynamicLayoutVariantSwitch)
	    {
        SetRect(aRect);
        ScaleImages();

        // adjust sprite positions if needed (new resolution may have caused
        // current position to be out of bounds
    	for ( TInt count = 0; count < iSprites->Count(); ++count )
    		{
            CSprite *sprite = iSprites->At(count);
    		sprite->Adjust( sprite->Image()->SizeInPixels(), Rect() );
    		}
	    }
*/	    
    }

// -----------------------------------------------------------------------------
// Scales background and sprite images according to current layout.
// -----------------------------------------------------------------------------
void CSymbiMazeAppView::ScaleImages()
    {
    AknIconUtils::SetSize(iBackgroundImage,
                          fieldSize,
                          EAspectRatioNotPreserved);

    AknIconUtils::SetSize(iBallImage, TSize(ballr*2+1, ballr*2+1));
    AknIconUtils::SetSize(iHoleImage, TSize(holer*2+1, holer*2+1));
    AknIconUtils::SetSize(iFinImage, TSize(holer*2+1, holer*2+1));

	delete iOffScreenBitmapGc;
	iOffScreenBitmapGc = NULL;
	delete iOffScreenBitmapDevice;
	iOffScreenBitmapDevice = NULL;
	delete iOffScreenBitmap;
	iOffScreenBitmap = NULL;

    // Create the off screen bitmap and device / gc
//    iOffScreenBitmap = NBitmapMethods::CreateBitmapL( iFullRect.Size(), KColourDepth );
//    iOffScreenBitmapDevice = NBitmapMethods::CreateBitmapDeviceL( *iOffScreenBitmap );
//    iOffScreenBitmapGc = NBitmapMethods::CreateSymbiMazeContextL( *iOffScreenBitmapDevice );
    }

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::UpdateDisplay()
// Updates the display.
// -----------------------------------------------------------------------------
//
void CSymbiMazeAppView::UpdateDisplay() const
{
    CWindowGc& gc = SystemGc();

    gc.BitBlt( shiftImg,iBackgroundImage );
//    iOffScreenBitmapGc->BitBlt( shiftImg, iBackgroundImage );

	TRgb col;
	TRect rect;
	
    col.SetRed(120); col.SetGreen(120); col.SetBlue(120);
    gc.SetBrushColor(col);
    gc.SetPenColor(col);
	if (shiftImg.iY>0)
	{
		rect.SetRect(
				TPoint(0,0),
				TSize(fieldSize.iWidth, shiftImg.iY)
				);
		gc.Clear(rect);
		rect.SetRect(
				TPoint(0,shiftImg.iY+fieldSize.iHeight),
				TSize(fieldSize.iWidth, shiftImg.iY)
				);
		gc.Clear(rect);
	}

	if (shiftImg.iX>0)
	{
		rect.SetRect(
				TPoint(0,0),
				TSize(shiftImg.iX, fieldSize.iHeight)
				);
		gc.Clear(rect);
		rect.SetRect(
				TPoint(shiftImg.iX+fieldSize.iWidth, 0),
				TSize(shiftImg.iX, fieldSize.iHeight)
				);
		gc.Clear(rect);
	}

	
    TInt lvlno = maze->cur_level;
    Level *lvl = &maze->qt_game_levels[lvlno];
    
	for (TInt i = 0; i<lvl->boxes_count; i++)
	{
		Box *box = &lvl->boxes[i];
		rect.SetRect(
				TRANS_X box->x1, TRANS_Y box->y1,
				TRANS_X box->x2, TRANS_Y box->y2
				);
		gc.Clear(rect);
//		iOffScreenBitmapGc->Clear(rect);
	}
	
	for (TInt i = 0; i<lvl->holes_count; i++)
	{
		Point *po = &lvl->holes[i];
		iHoleSprite->SetPosition(TPoint(
				        TRANS_X (po->x - maze->qt_game_config.hole_r), 
						TRANS_Y (po->y - maze->qt_game_config.hole_r)
						));
		TRect sourceRectHole( TPoint( 0,0 ), iHoleSprite->Image()->SizeInPixels() );
		gc.BitBltMasked( 
						iHoleSprite->Position(),
						iHoleSprite->Image(),
						sourceRectHole,
						iHoleSprite->Mask(),
						ETrue
						);
//        NBitmapMethods::BitBltMaskedEntireBitmap( 
//						*iOffScreenBitmapGc,
//						iHoleSprite->Position(),
//						*(iHoleSprite->Image()),
//						*(iHoleSprite->Mask()) );		
	}	

	Point *po = &lvl->fins[0];
	iFinSprite->SetPosition(TPoint(
			        TRANS_X (po->x - maze->qt_game_config.hole_r), 
			        TRANS_Y (po->y - maze->qt_game_config.hole_r)
					));
	TRect sourceRectFin( TPoint( 0,0 ), iFinSprite->Image()->SizeInPixels() );
	gc.BitBltMasked( 
					iFinSprite->Position(),
					iFinSprite->Image(),
					sourceRectFin,
					iFinSprite->Mask(),
					ETrue
					);
//    NBitmapMethods::BitBltMaskedEntireBitmap( 
//					*iOffScreenBitmapGc,
//					iFinSprite->Position(),
//					*(iFinSprite->Image()),
//					*(iFinSprite->Mask()) );		
	
	iBallSprite->SetPosition(TPoint(
					TRANS_X (maze->px - maze->qt_game_config.ball_r),
					TRANS_Y (maze->py - maze->qt_game_config.ball_r)
					));
	TRect sourceRectBall( TPoint( 0,0 ), iBallSprite->Image()->SizeInPixels() );
	gc.BitBltMasked( 
					iBallSprite->Position(),
					iBallSprite->Image(),
					sourceRectBall,
					iBallSprite->Mask(),
					ETrue
					);
//    NBitmapMethods::BitBltMaskedEntireBitmap( 
//					*iOffScreenBitmapGc,
//					iBallSprite->Position(),
//					*(iBallSprite->Image()),
//					*(iBallSprite->Mask()) );	

	if (!game)
	{
		_LIT(KLevelFormatter, "Level %d/%d");
		TBuf<255> tLevelBuf;	
		tLevelBuf.Format(KLevelFormatter, maze->cur_level+1, maze->qt_game_levels_count);

		col.SetRed(114); col.SetGreen(190); col.SetBlue(231);
		gc.SetBrushColor(col);
		gc.SetPenColor(col);
		
		const CFont *font=iEikonEnv->TitleFont();
		gc.UseFont(font);
		int tLevelWidth = font->TextWidthInPixels(tLevelBuf);
		gc.DrawText(tLevelBuf, shiftImg + TPoint((fieldSize.iWidth-tLevelWidth)/2, fieldSize.iHeight/4));
		gc.DiscardFont();
	}
    
//    gc.BitBlt( Rect().iTl,iOffScreenBitmap );    
}

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::DoPeriodTask()
// Move the sprites.
// -----------------------------------------------------------------------------
//
void CSymbiMazeAppView::DoPeriodTask()
	{
	maze->tout();
	ForceRedraw();
	User::ResetInactivityTime();
	}

void CSymbiMazeAppView::ForceRedraw()
{
	// Update the screen
	CWindowGc& gc = SystemGc();
	gc.Activate( *DrawableWindow() );
	UpdateDisplay();
	gc.Deactivate();
}

// -----------------------------------------------------------------------------
// CSymbiMazeAppView::Period()
// This function is called by the periodic timer.
// -----------------------------------------------------------------------------
//
TInt CSymbiMazeAppView::Period( TAny* aPtr )
	{
    ( static_cast<CSymbiMazeAppView*>( aPtr ) )->DoPeriodTask();
    // Returning a value of TRUE indicates the callback should be done again
	return ETrue;
	}

//******************************************************************************
//******************************************************************************

void CSymbiMazeAppView::HandlePointerEventL(const TPointerEvent& aPointerEvent)
	{
	if (aPointerEvent.iType == TPointerEvent::EButton1Down) ToggleGame();
	
	// Call base class HandlePointerEventL()
	CCoeControl::HandlePointerEventL(aPointerEvent);
	}


TKeyResponse CSymbiMazeAppView::OfferKeyEventL( const TKeyEvent& aKeyEvent, TEventCode aType )	 
{
	if( aType == EEventKeyDown )	 
	{	 
	  // Process EEventKeyDown	 
	}	 
	 
	if( aType == EEventKey )	 
	{	 
	  TInt sc = aKeyEvent.iScanCode;
	  if ( (sc==EStdKeyDevice3) ||
		   ((sc>=48) && (sc<=57)) )
		  {
			  ToggleGame();
			  return EKeyWasConsumed;
		  }
	  
	  if (!game)
	  {
		  if (sc==EStdKeyLeftArrow) 
		  {
			  PrevLevel();
			  return EKeyWasConsumed;
		  }
		  if (sc==EStdKeyRightArrow) 
		  {
			  NextLevel();
			  return EKeyWasConsumed;
		  }
	  }
	}	 
	 
	if( aType == EEventKeyUp )	 
	{
	  // Process EEventKeyUp 	 
	}	 

	return EKeyWasNotConsumed;
}

void CSymbiMazeAppView::PrevLevel()
{
	if (maze->cur_level > 0)
	{
		maze->cur_level--;
		maze->InitState();
		maze->game_state = GAME_STATE_NORMAL;
		ForceRedraw();
	}
}

void CSymbiMazeAppView::NextLevel()
{
	if (maze->cur_level < maze->qt_game_levels_count-1)
	{
		maze->cur_level++;
		maze->InitState();
		maze->game_state = GAME_STATE_NORMAL;
		ForceRedraw();
	}
}

void CSymbiMazeAppView::ResetLevel()
{
	maze->cur_level = 0;
	maze->InitState();
	maze->game_state = GAME_STATE_NORMAL;
	ForceRedraw();
}

void CSymbiMazeAppView::FocusEvent(TBool isFocused)
	{
		if (!isFocused)
			StopGame();
	}

void CSymbiMazeAppView::StartGame()
	{
		game=ETrue;
		SetRect( iFullRect );
		if ( !iPeriodicTimer->IsActive() )
		{
			iPeriodicTimer->Start(1, PROC_ACC_DATA_INTERVAL*1000, TCallBack( CSymbiMazeAppView::Period, this));
		}
	}

void CSymbiMazeAppView::StopGame()
	{
		game=EFalse;
		SetRect( iAppRect );
		if ( iPeriodicTimer->IsActive() )
		{
			iPeriodicTimer->Cancel();
		}
		ForceRedraw();
	}

void CSymbiMazeAppView::ToggleGame()
	{
		if (game)
			StopGame();
		else
			StartGame();
	}

void CSymbiMazeAppView::AccelerometerData( TReal aX, TReal aY )
    {
		if (aX<-1) aX=-1; if (aX>1) aX=1;
		if (aY<-1) aY=-1; if (aY>1) aY=1;
		maze->getacx = -aX;
		maze->getacy = -aY;
    }

void CSymbiMazeAppView::InitMaze()
{
#ifdef DEBUG_ENGINE    
    _LIT(debug_levelpack,
    "{"
    ""
    "\"name\":   \"Test\","
    "\"author\": \"ANT\","
    ""
    "\"requirements\": {"
    ""
    "    \"window\": {"
    "        \"width\":  480,"
    "        \"height\": 640"
    "    },"
    ""
    "    \"ball\": {"
    "        \"radius\": 23"
    "    },"
    ""
    "    \"hole\": {"
    "        \"radius\": 28"
    "    }"
    ""
    "},"
    ""
    "\"levels\": ["
    "	{"
    ""
    "	    \"comment\": \"level 1\","
    ""
    "	    \"boxes\": ["
    "	        { \"x1\": 150,	\"y1\": 110,	\"x2\": 160,	\"y2\": 639 },"
    "	        { \"x1\": 320,	\"y1\": 0,	\"x2\": 330,	\"y2\": 529 }"
    "	    ],"
    ""
    "	    \"holes\": ["
    "	        { \"x\": 289, 	\"y\": 199 },"
    "	        { \"x\": 191, 	\"y\": 427 }"
    "	    ],"
    ""
    "	    \"checkpoints\": ["
    "	        { \"x\": 410, 	\"y\": 65 }"
    "	    ],"
    ""
    "	    \"init\": {"
    "	        \"x\": 77, 	\"y\": 567"
    "	    }"
    ""
    "	},"
    ""
    "	{"
    ""
    "	    \"comment\": \"level 2\","
    ""
    "	    \"boxes\": ["
    "	        { \"x1\": 70,	\"y1\": 80,	\"x2\": 410,	\"y2\": 90 }"
    "	    ],"
    ""
    "	    \"holes\": ["
    "	        { \"x\": 31, 	\"y\": 314 },"
    "	        { \"x\": 212, 	\"y\": 314 },"
    "	        { \"x\": 92, 	\"y\": 314 },"
    "	        { \"x\": 152, 	\"y\": 314 },"
    "	        { \"x\": 447, 	\"y\": 314 },"
    "	        { \"x\": 386, 	\"y\": 314 },"
    "	        { \"x\": 327, 	\"y\": 314 }"
    "	    ],"
    ""
    "	    \"checkpoints\": ["
    "	        { \"x\": 76, 	\"y\": 403 }"
    "	    ],"
    ""
    "	    \"init\": {"
    "	        \"x\": 245, 	\"y\": 42"
    "	    }"
    ""	
    "	}"
    "]"
    "}"
    "" );    
#endif    

#ifndef DEBUG_ENGINE    
    RFile file;
    TInt err;
    err = file.Open( CEikonEnv::Static()->FsSession(), _L("c:\\private\\" APP_UID_STR "\\main.levelpack.json"), EFileRead );
    if (err != KErrNone)
    {
        err = file.Open( CEikonEnv::Static()->FsSession(), _L("e:\\private\\" APP_UID_STR "\\main.levelpack.json"), EFileRead );
    }
    //TODO: IO error check
    
    TInt fsize;
    file.Size(fsize);
    TUint8* fileBuf = (TUint8*)User::AllocL(fsize*sizeof(TUint8));
    TPtr8 fileBufPtr(fileBuf, fsize);
    file.Read(fileBufPtr);
    file.Close();
    
    TUint16* fileBuf16 = (TUint16*)User::AllocL(fsize*sizeof(TUint16));
    TPtr16 fileBufPtr16(fileBuf16, fsize);
    fileBufPtr16.Copy(fileBufPtr);
    
    delete fileBuf;
#endif    

	CJsonParser parser;
#ifndef DEBUG_ENGINE    
    parser.StartDecodingL(fileBufPtr16);
    delete fileBuf16;
#else
    parser.StartDecodingL(debug_levelpack);
#endif    

    int bufSize = 32; //
    TUint16* buf = (TUint16*)User::AllocL(bufSize*sizeof(TUint16));
    TPtr16 bufPtr(buf, bufSize);

    maze = CGame::NewL();
    
	parser.GetParameterValue(_L("[requirements][window][width]"), &bufPtr);
    maze->qt_game_config.wnd_w = GetInt(bufPtr);
	parser.GetParameterValue(_L("[requirements][window][height]"), &bufPtr);
	maze->qt_game_config.wnd_h = GetInt(bufPtr);
	parser.GetParameterValue(_L("[requirements][ball][radius]"), &bufPtr);
	maze->qt_game_config.ball_r = GetInt(bufPtr);
	parser.GetParameterValue(_L("[requirements][hole][radius]"), &bufPtr);
	maze->qt_game_config.hole_r = GetInt(bufPtr);

    _LIT(LevBoxCountFormatter, "[levels][%d][boxes]");
    _LIT(LevBoxX1Formatter, "[levels][%d][boxes][%d][x1]");
    _LIT(LevBoxX2Formatter, "[levels][%d][boxes][%d][x2]");
    _LIT(LevBoxY1Formatter, "[levels][%d][boxes][%d][y1]");
    _LIT(LevBoxY2Formatter, "[levels][%d][boxes][%d][y2]");

    _LIT(LevHoleCountFormatter, "[levels][%d][holes]");
    _LIT(LevHoleXFormatter, "[levels][%d][holes][%d][x]");
    _LIT(LevHoleYFormatter, "[levels][%d][holes][%d][y]");

    _LIT(LevFinXFormatter, "[levels][%d][checkpoints][0][x]");
    _LIT(LevFinYFormatter, "[levels][%d][checkpoints][0][y]");

    _LIT(LevInitXFormatter, "[levels][%d][init][x]");
    _LIT(LevInitYFormatter, "[levels][%d][init][y]");
    
    int levco = parser.GetParameterCount(_L("[levels]"));
    maze->qt_game_levels_count = levco;
    maze->qt_game_levels = (Level*)User::AllocL(levco*sizeof(Level));

    TBuf<255> tbuf;
    for (TInt i=0; i<levco; i++)
	{
    	Level *lvl = &maze->qt_game_levels[i];
    	int co; 
    	
    	//boxes
		tbuf.Format(LevBoxCountFormatter, i);
	    co = parser.GetParameterCount(tbuf);
	    lvl->boxes_count = co;
		lvl->boxes = (Box*)User::AllocL(co*sizeof(Box));
	    for (TInt j=0; j<co; j++)
		{
			Box b;
			
			tbuf.Format(LevBoxX1Formatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			b.x1 = GetInt(bufPtr);
			
			tbuf.Format(LevBoxY1Formatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			b.y1 = GetInt(bufPtr);

			tbuf.Format(LevBoxX2Formatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			b.x2 = GetInt(bufPtr);

			tbuf.Format(LevBoxY2Formatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			b.y2 = GetInt(bufPtr);
			
			lvl->boxes[j] = b;
		}

    	//holes
		tbuf.Format(LevHoleCountFormatter, i);
	    co = parser.GetParameterCount(tbuf);
	    lvl->holes_count = co;
		lvl->holes = (Point*)User::AllocL(co*sizeof(Point));
	    for (TInt j=0; j<co; j++)
		{
			Point p;
			
			tbuf.Format(LevHoleXFormatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			p.x = GetInt(bufPtr);
			
			tbuf.Format(LevHoleYFormatter, i,j);
			parser.GetParameterValue(tbuf, &bufPtr);
			p.y = GetInt(bufPtr);
			
			lvl->holes[j] = p;
		}

		Point fi;		
		tbuf.Format(LevFinXFormatter, i);
		parser.GetParameterValue(tbuf, &bufPtr);
		fi.x = GetInt(bufPtr);
		tbuf.Format(LevFinYFormatter, i);
		parser.GetParameterValue(tbuf, &bufPtr);
		fi.y = GetInt(bufPtr);		
		lvl->fins_count = 1;
		lvl->fins[0] = fi;

		Point init;		
		tbuf.Format(LevInitXFormatter, i);
		parser.GetParameterValue(tbuf, &bufPtr);
		init.x = GetInt(bufPtr);		
		tbuf.Format(LevInitYFormatter, i);
		parser.GetParameterValue(tbuf, &bufPtr);
		init.y = GetInt(bufPtr);		
		lvl->init = init;
	}
    
//------------------------------------------------------------------------------
    maze->cur_level = 0;
    
    saveLoaded = EFalse;
    saveOnE = EFalse;    
    
#ifndef DEBUG_ENGINE     
    RFile saveFile;
    TInt saveErr;
    
    saveErr = saveFile.Open( CEikonEnv::Static()->FsSession(), _L("c:\\private\\" APP_UID_STR "\\user.json"), EFileRead );
    if (saveErr != KErrNone)
    {
        saveErr = saveFile.Open( CEikonEnv::Static()->FsSession(), _L("e:\\private\\" APP_UID_STR "\\user.json"), EFileRead );
        saveOnE = ETrue;
    }
    
    if (saveErr == KErrNone)
    {
        TInt saveFileSize;
        saveFile.Size(saveFileSize);
        TUint8* saveFileBuf = (TUint8*)User::AllocL(saveFileSize*sizeof(TUint8));
        TPtr8 saveFileBufPtr(saveFileBuf, saveFileSize);
        saveFile.Read(saveFileBufPtr);
        saveFile.Close();
        
        TUint16* saveFileBuf16 = (TUint16*)User::AllocL(saveFileSize*sizeof(TUint16));
        TPtr16 saveFileBufPtr16(saveFileBuf16, saveFileSize);
        saveFileBufPtr16.Copy(saveFileBufPtr);
        
        delete saveFileBuf;
    
        CJsonParser saveParser;
        saveParser.StartDecodingL(saveFileBufPtr16);
        delete saveFileBuf16;
    
        int saveBufSize = 32; //
        TUint16* saveBuf = (TUint16*)User::AllocL(saveBufSize*sizeof(TUint16));
        TPtr16 saveBufPtr(saveBuf, saveBufSize);
        
        saveParser.GetParameterValue(_L("[save][level]"), &saveBufPtr);
        maze->cur_level = GetInt(saveBufPtr) - 1;
        
        if (maze->cur_level < 0)
            maze->cur_level = 0;
        if (maze->cur_level > maze->qt_game_levels_count-1)
            maze->cur_level = maze->qt_game_levels_count-1;
        
        saveLoaded = ETrue;
    }
#endif
//------------------------------------------------------------------------------    

    maze->game_state = GAME_STATE_NORMAL;
    maze->InitState();
    
    maze->InitVibro();
}

TInt CSymbiMazeAppView::GetInt(TPtr16 &sp)
{
	TInt res;
	TLex lex(sp);    
    lex.Val(res);
    return res;
}

void CSymbiMazeAppView::PrepToQuit()
{
    if ( iPeriodicTimer->IsActive() )
    {
        iPeriodicTimer->Cancel();
    }
    maze->StopVibro();
}

void CSymbiMazeAppView::SaveState()
{
#ifndef DEBUG_ENGINE
    if (saveLoaded)
    {
        _LIT(SaveTemplate, "{\n\t\"save\": {\n\t\t\"levelpack\": \"main\",\n\t\t\"level\"    : %d\n\t}\n}\n");
        RFile file;
        TInt err;
        if (!saveOnE)
            err = file.Replace( CEikonEnv::Static()->FsSession(), _L("c:\\private\\" APP_UID_STR "\\user.json"), EFileWrite );
        else    
            err = file.Replace( CEikonEnv::Static()->FsSession(), _L("e:\\private\\" APP_UID_STR "\\user.json"), EFileWrite );
        if (err == KErrNone)
        {
            TBuf<512> tbuf;
            tbuf.Format(SaveTemplate, maze->cur_level + 1);
            TBuf8<512> tbuf8;
            tbuf8.Copy(tbuf);
            file.Write(tbuf8);
            file.Close();
        }
    }
#endif
}

void CSymbiMazeAppView::MrccatoCommand(TRemConCoreApiOperationId aOperationId,
        TRemConCoreApiButtonAction aButtonAct)
{
    switch (aOperationId)
    {
    case ERemConCoreApiVolumeDown:
    {
        PrevLevel();
        break;
    }
    case ERemConCoreApiVolumeUp:
    {
        NextLevel();
        break;
    }
    }
}

void CSymbiMazeAppView::ShowHelpDlg()
{
    _LIT(KHelpTitle, "Help");
    _LIT(KHelpText,
            "Control the ball by tilting your accelerometer-enabled smartphone. "
            "Reach the exit avoiding other holes." "\n"            
            "----" "\n"
            "Press Select key or touch the screen to pause/resume the game. "
            "Use left/right or volume keys to switch levels.");
    
    TPtrC title = KHelpTitle();    
    TPtrC text = KHelpText();    
    CAknMessageQueryDialog* dlg = CAknMessageQueryDialog::NewL(text);
    dlg->PrepareLC(R_ABOUT_DIALOG);
    dlg->SetHeaderTextL(title);
    dlg->RunLD();    
}

void CSymbiMazeAppView::ShowAboutDlg()
{
    _LIT(KAboutTitle, "About");
    _LIT(KAboutText,
            "SymbiMaze 0.10" "\n"
            "(C) 2009 Anton Olkhovik" "\n"
            "http://mokomaze.projects.openmoko.org/symbian/" "\n"
            "----" "\n"
            "This program is free software: you can redistribute it and/or modify "
            "it under the terms of the GNU General Public License as published by "
            "the Free Software Foundation, either version 3 of the License, or "
            "(at your option) any later version.");
    
    TPtrC title = KAboutTitle();    
    TPtrC text = KAboutText();    
    CAknMessageQueryDialog* dlg = CAknMessageQueryDialog::NewL(text);
    dlg->PrepareLC(R_ABOUT_DIALOG);
    dlg->SetHeaderTextL(title);
    dlg->RunLD();    
}

// End of File
