/*  game.cpp
 *
 *  Main game routines
 *
 *  (c) 2009 Anton Olkhovik <ant007h@gmail.com>
 *
 *  This file is part of SymbiMaze (port of Mokomaze) - labyrinth game.
 *
 *  SymbiMaze is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  SymbiMaze is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SymbiMaze.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <e32math.h>
#include "game.h"

CGame::CGame(CSymbiMazeAppView* aAppView) :
    iVibra(NULL)
    , iAppView(aAppView)
{
    // No implementation required
}

CGame* CGame::NewL(CSymbiMazeAppView* aAppView)
{
    CGame* self = NewLC(aAppView);
    CleanupStack::Pop(self);
    return self;
}

CGame* CGame::NewLC(CSymbiMazeAppView* aAppView)
{
    CGame* self = new (ELeave) CGame(aAppView);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
}

CGame::~CGame()
{
    delete iVibra;
}

void CGame::ConstructL()
{
    // No implementation required
}

//******************************************************************************

TReal CGame::fabs(TReal x)
{
    if (x < 0)
        return -x;
    else
        return x;
}

TReal CGame::sqrt(TReal x)
{
    TReal res;
    Math::Sqrt(res, x);
    return res;
}

TReal CGame::cos(TReal x)
{
    TReal res;
    Math::Cos(res, x);
    return res;
}

TReal CGame::asin(TReal x)
{
    TReal res;
    Math::ASin(res, x);
    return res;
}

bool CGame::InRect(int px, int py, int x, int y, int w, int h)
{
    return ((px >= x) && (px <= x + w) && (py >= y) && (py <= y + h));
}

void CGame::BumpVibrate(TReal speed)
{
    TReal v = fabs(speed);
    TReal k = v / MAX_BUMP_SPEED;
    if (k > 1)
        k = 1;
    if (v > MIN_BUMP_SPEED)
    {
        TUint8 vlevel = (TUint8) (k * 100);
        //set_vibro(vlevel);
        if (iVibra != NULL)
            if (iVibra->VibraStatus() != CHWRMVibra::EVibraStatusNotAllowed)
            {
                iVibra->VibraStartL(VIBRATION_TIME, vlevel);
            }
    }
}

void CGame::post_phys_res(TReal x, TReal y, TReal mm_vx, TReal mm_vy)
{
    px = x;
    py = y;
    MoveBall(px, py); //

    vx = mm_vx;
    vy = mm_vy;
    pr_px = px;
    pr_py = py;
    pr_vx = vx;
    pr_vy = vy;
}

void CGame::post_temp_phys_res(TReal x, TReal y, TReal mm_vx, TReal mm_vy)
{
    if (x < qt_game_config.ball_r)
    {
        BumpVibrate(mm_vx); //VIB_HOR
        x = qt_game_config.ball_r;
        mm_vx = -mm_vx * BUMP_COEF;
    }
    if (x > qt_game_config.wnd_w - qt_game_config.ball_r)
    {
        BumpVibrate(mm_vx);
        x = qt_game_config.wnd_w - qt_game_config.ball_r;
        mm_vx = -mm_vx * BUMP_COEF;
    }
    if (y < qt_game_config.ball_r)
    {
        BumpVibrate(mm_vy);
        y = qt_game_config.ball_r;
        mm_vy = -mm_vy * BUMP_COEF;
    }
    if (y > qt_game_config.wnd_h - qt_game_config.ball_r)
    {
        BumpVibrate(mm_vy);
        y = qt_game_config.wnd_h - qt_game_config.ball_r;
        mm_vy = -mm_vy * BUMP_COEF;
    }

    tmp_px = x;
    tmp_py = y;
    tmp_vx = mm_vx;
    tmp_vy = mm_vy;
}

void CGame::apply_temp_phys_res()
{
    post_phys_res(tmp_px, tmp_py, tmp_vx, tmp_vy);
}

void CGame::tout()
{
    //printf("[acc] %.4f %.4f %.4f\n", getacx(), getacy(), getacz());

    if (game_state != GAME_STATE_NORMAL)
        return;
    new_game_state = GAME_STATE_NORMAL;

    TReal ax = getacx;
    TReal ay = getacy;

    TReal mid_px = px, mid_py = py;
    TReal mid_vx = vx, mid_vy = vy;

    TReal v = sqrt(mid_vx * mid_vx + mid_vy * mid_vy);
    TReal a = sqrt(ax * ax + ay * ay);
    if ((v > 0) || (a > FORCE_TREASURE))
    {
        mid_vx += ax * GRAV_CONST * TIME_QUANT;
        mid_vy += ay * GRAV_CONST * TIME_QUANT;
    }
    if (fabs(mid_vx) > 0)
    {
        TReal dvx = fabs(FRICT_COEF * GRAV_CONST * cos(asin(ax)));
        if (mid_vx > 0)
        {
            mid_vx -= dvx;
            if (mid_vx < 0)
                mid_vx = 0;
        }
        else
        {
            mid_vx += dvx;
            if (mid_vx > 0)
                mid_vx = 0;
        }
    }
    if (fabs(mid_vy) > 0)
    {
        TReal dvy = fabs(FRICT_COEF * GRAV_CONST * cos(asin(ay)));
        if (mid_vy > 0)
        {
            mid_vy -= dvy;
            if (mid_vy < 0)
                mid_vy = 0;
        }
        else
        {
            mid_vy += dvy;
            if (mid_vy > 0)
                mid_vy = 0;
        }
    }

    mid_px += (mid_vx * SPEED_TO_PIXELS);
    mid_py += -(mid_vy * SPEED_TO_PIXELS);

    if (!line(pr_px, pr_py, mid_px, mid_py, pr_vx, pr_vy, mid_vx, mid_vy))
    {
        post_temp_phys_res(mid_px, mid_py, mid_vx, mid_vy);
        apply_temp_phys_res();
    }
}

TReal CGame::calcdist(TReal x1, TReal y1, TReal x2, TReal y2)
{
    return sqrt((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));
}

TReal CGame::calclen(TReal x, TReal y)
{
    return sqrt(x * x + y * y);
}

DPoint CGame::normalize(DPoint p)
{
    DPoint r;
    TReal len = calclen(p.x, p.y);
    r.x = p.x / len;
    r.y = p.y / len;
    return r;
}

int CGame::inbox(TReal x, TReal y, Box box)
{
    return ((x >= box.x1) && (x <= box.x2) && (y >= box.y1) && (y <= box.y2));
}

int CGame::incircle(TReal x, TReal y, TReal cx, TReal cy, TReal cr)
{
    return (calcdist(x, y, cx, cy) <= cr);
}

int CGame::edgebump(int tx, int ty, TReal x, TReal y, TReal mm_vx, TReal mm_vy)
{
    Point touch;
    touch.x = tx;
    touch.y = ty;
    if (incircle(touch.x, touch.y, x, y, qt_game_config.ball_r))
    {
        DPoint tmp_norm;
        tmp_norm.x = (x - touch.x);
        tmp_norm.y = (y - touch.y);
        DPoint norm = normalize(tmp_norm);
        mm_vy = -mm_vy;
        TReal dot_norm_vect = mm_vx * norm.x + mm_vy * norm.y;
        mm_vx = mm_vx - 2 * norm.x * dot_norm_vect;
        mm_vy = mm_vy - 2 * norm.y * dot_norm_vect;

        TReal vpr_x = -fabs(dot_norm_vect) * norm.x;
        TReal vpr_y = -fabs(dot_norm_vect) * norm.y;
        BumpVibrate(calclen(vpr_x, vpr_y));

        TReal dop_x = (1 - BUMP_COEF) * vpr_x;
        TReal dop_y = (1 - BUMP_COEF) * vpr_y;
        mm_vx = mm_vx + dop_x;
        mm_vy = mm_vy + dop_y;
        mm_vy = -mm_vy;

        x = touch.x + norm.x * (qt_game_config.ball_r + 0.10);
        y = touch.y + norm.y * (qt_game_config.ball_r + 0.10);

        post_temp_phys_res(x, y, mm_vx, mm_vy);
        return 1;
    }
    return 0;
}

int CGame::testbump(TReal x, TReal y, TReal mm_vx, TReal mm_vy)
{

    Point final_hole = qt_game_levels[cur_level].fins[0];
    TReal dist = calcdist(x, y, final_hole.x, final_hole.y);
    if (dist <= qt_game_config.hole_r)
    {
        hole_x = final_hole.x;
        hole_y = final_hole.y;
        new_game_state = GAME_STATE_WIN;
        post_temp_phys_res(x, y, mm_vx, mm_vy);
        return 1;
    }

    for (int i = 0; i < qt_game_levels[cur_level].holes_count; i++)
    {
        Point hole = qt_game_levels[cur_level].holes[i];
        Box boundbox;
        boundbox.x1 = hole.x - qt_game_config.hole_r - 1;
        boundbox.y1 = hole.y - qt_game_config.hole_r - 1;
        boundbox.x2 = hole.x + qt_game_config.hole_r + 1;
        boundbox.y2 = hole.y + qt_game_config.hole_r + 1;
        if (inbox(x, y, boundbox))
        {
            TReal dist = calcdist(x, y, hole.x, hole.y);
            if (dist <= qt_game_config.hole_r)
            {
                hole_x = hole.x;
                hole_y = hole.y;
                new_game_state = GAME_STATE_FAILED;
                post_temp_phys_res(x, y, mm_vx, mm_vy);
                return 1;
            }
        }
    }

    int retval = 0;

    for (int i = 0; i < qt_game_levels[cur_level].boxes_count; i++)
    {
        Box box = qt_game_levels[cur_level].boxes[i];
        Box boundbox;
        boundbox.x1 = box.x1 - qt_game_config.ball_r - 1; //
        boundbox.y1 = box.y1 - qt_game_config.ball_r - 1;
        boundbox.x2 = box.x2 + qt_game_config.ball_r + 1;
        boundbox.y2 = box.y2 + qt_game_config.ball_r + 1;
        if (inbox(x, y, boundbox))
        {
            if (inbox(x, y - qt_game_config.ball_r, box))
            {
                BumpVibrate(mm_vy);
                y = box.y2 + qt_game_config.ball_r + 0.2;
                mm_vy = -mm_vy * BUMP_COEF;
                retval = 1;
            }
            if (inbox(x, y + qt_game_config.ball_r - 0, box))
            {
                BumpVibrate(mm_vy);
                y = box.y1 - qt_game_config.ball_r - 1.00;
                mm_vy = -mm_vy * BUMP_COEF;
                retval = 1;
            }
            if (inbox(x - qt_game_config.ball_r, y, box))
            {
                BumpVibrate(mm_vx);
                x = box.x2 + qt_game_config.ball_r + 0.2;
                mm_vx = -mm_vx * BUMP_COEF;
                retval = 1;
            }
            if (inbox(x + qt_game_config.ball_r - 0, y, box))
            {
                BumpVibrate(mm_vx);
                x = box.x1 - qt_game_config.ball_r - 1.00;
                mm_vx = -mm_vx * BUMP_COEF;
                retval = 1;
            }

            if (edgebump(box.x1, box.y1, x, y, mm_vx, mm_vy))
                return 1;
            if (edgebump(box.x2, box.y1, x, y, mm_vx, mm_vy))
                return 1;
            if (edgebump(box.x2, box.y2, x, y, mm_vx, mm_vy))
                return 1;
            if (edgebump(box.x1, box.y2, x, y, mm_vx, mm_vy))
                return 1;
        }

    }

    if (retval)
    {
        post_temp_phys_res(x, y, mm_vx, mm_vy);
    }

    return retval;
}

int CGame::line(TReal x0, TReal y0, TReal x1, TReal y1, TReal vx0, TReal vy0,
        TReal vx1, TReal vy1)
{

    TReal x = x0, y = y0;
    TReal mm_vx = vx1, mm_vy = vy1; //

    DPoint vec;
    vec.x = x1 - x0;
    vec.y = y1 - y0;
    DPoint norm_vec;
    TReal len = calclen(vec.x, vec.y);

    norm_vec.x = vec.x / len;
    norm_vec.y = vec.y / len;
    TReal k = 0;
    int muststop = 0;
    while (1)
    {
        k += 1;
        if (k >= len)
        {
            k = len;
            muststop = 1;
        }
        x = x0 + norm_vec.x * k;
        y = y0 + norm_vec.y * k;
        if (testbump(x, y, mm_vx, mm_vy))
        {
            int ko = 0;
            while (new_game_state == GAME_STATE_NORMAL)
            {
                ko++;
                //TODO: lite testbump version
                int bump = testbump(tmp_px, tmp_py, tmp_vx, tmp_vy);
                if ((!bump) || (ko >= MAX_PHYS_ITERATIONS))
                    break;
            }
            apply_temp_phys_res();
            game_state = new_game_state;
            //ProcessGameState();
            return 1;
        }
        if (muststop)
            break;
    }

    return 0;

}

void CGame::InitState()
{
    //renderArea->setLevel(cur_level);
    //renderArea->update();

    px = qt_game_levels[cur_level].init.x;
    py = qt_game_levels[cur_level].init.y;
    vx = 0;
    vy = 0;

    pr_px = px;
    pr_py = py;
    pr_vx = 0;
    pr_vy = 0;

    prev_px = px;
    prev_py = py;
    MoveBall(px, py); //
}

void CGame::MoveBall(TReal x, TReal y)
{
    //ball_lbl->move((int)x-qt_game_config.ball_r, (int)y-qt_game_config.ball_r);
}

void CGame::InitVibro()
{
    TRAPD( err, iVibra = CVibraController::NewL() );
    if (err == KErrNone) //KErrNotSupported
    {
        iVibra->VibraReserveL(EFalse, EFalse);
    }
}

void CGame::StopVibro()
{
    if (iVibra != NULL)
        iVibra->VibraStopL();
}

// End of File
